---
title: Reply to emails from Workers · Cloudflare Email Routing docs
description: You can reply to incoming emails with another new message and
  implement smart auto-responders programmatically, adding any content and
  context in the main body of the message. Think of a customer support email
  automatically generating a ticket and returning the link to the sender, an
  out-of-office reply with instructions when you are on vacation, or a detailed
  explanation of why you rejected an email.
lastUpdated: 2025-03-12T19:09:43.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/email-routing/email-workers/reply-email-workers/
  md: https://developers.cloudflare.com/email-routing/email-workers/reply-email-workers/index.md
---

You can reply to incoming emails with another new message and implement smart auto-responders programmatically, adding any content and context in the main body of the message. Think of a customer support email automatically generating a ticket and returning the link to the sender, an out-of-office reply with instructions when you are on vacation, or a detailed explanation of why you rejected an email.

Reply to emails is a new method of the [`EmailMessage` object](https://developers.cloudflare.com/email-routing/email-workers/runtime-api/#emailmessage-definition) in the Runtime API. Here is how it works:

```js
import { EmailMessage } from "cloudflare:email";
import { createMimeMessage } from "mimetext";


export default {
  async email(message, env, ctx) {


    const ticket = createTicket(message);


    const msg = createMimeMessage();
    msg.setHeader("In-Reply-To", message.headers.get("Message-ID"));
    msg.setSender({ name: "Thank you for your contact", addr: "<SENDER>@example.com" });
    msg.setRecipient(message.from);
    msg.setSubject("Email Routing Auto-reply");
    msg.addMessage({
      contentType: 'text/plain',
      data: `We got your message, your ticket number is ${ ticket.id }`
    });


    const replyMessage = new EmailMessage(
      "<SENDER>@example.com",
      message.from,
      msg.asRaw()
    );


    await message.reply(replyMessage);
  }
}
```

To mitigate security risks and abuse, replying to incoming emails has a few requirements and limits:

* The incoming email has to have valid [DMARC](https://www.cloudflare.com/learning/dns/dns-records/dns-dmarc-record/).
* The email can only be replied to once in the same `EmailMessage` event.
* The recipient in the reply must match the incoming sender.
* The outgoing sender domain must match the same domain that received the email.
* Every time an email passes through Email Routing or another MTA, an entry is added to the `References` list. We stop accepting replies to emails with more than 100 `References` entries to prevent abuse or accidental loops.

If these and other internal conditions are not met, `reply()` will fail with an exception. Otherwise, you can freely compose your reply message, send it back to the original sender, and receive subsequent replies multiple times.
